###LOAD DEPENDENT PACKAGES
library(xlsx)
library(rstatix)
library(outliers)
library(ggplot2)
library(ggpubr)

###FORMAT DATA
Data <- read.xlsx("Discrete_Kinematics.xlsx", sheetIndex = 1)
Data[,4:23] <- abs(Data[,4:23])
Hip <- Data[,c(1,4:8)]

###OUTLIER IDENTIFICATION

Ay.Hip <- Hip[1:30,]
Ma.Hip <- Hip[31:60,]
Ir.Hip <- Hip[61:90,]

#Grubb's test to identify outliers
#Aylesburys
for(i in 2:ncol(Ay.Hip)){
  DV <- names(Ay.Hip[i])
  res.grub <- grubbs.test(Ay.Hip[,i], type = 11) #test if highest and lowest values are outliers
  print(DV)
  print(res.grub)
}
#Mallards
for(i in 2:ncol(Ma.Hip)){
  DV <- names(Ma.Hip[i])
  res.grub <- grubbs.test(Ma.Hip[,i], type = 11) #test if highest and lowest values are outliers
  print(DV)
  print(res.grub)
}
#Runners
for(i in 2:ncol(Ir.Hip)){
  DV <- names(Ir.Hip[i])
  res.grub <- grubbs.test(Ir.Hip[,i], type = 11) #test if highest and lowest values are outliers
  print(DV)
  print(res.grub)
}

#Boxplots to investigation data distribution

Box.Hip.Max <- ggplot(Hip, aes(x=Breed, y=Hip_Max_Flex, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Maximum Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Box.Hip.Min <- ggplot(Hip, aes(x=Breed, y=Hip_Min_Flex, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Minimum Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Box.Toe.On <- ggplot(Hip, aes(x=Breed, y=Hip_Toe_on, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Toe-on Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Box.Toe.Off <- ggplot(Hip, aes(x=Breed, y=Hip_Toe_off, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Toe-off Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Box.Mid.Stance <- ggplot(Hip, aes(x=Breed, y=Hip_Mid_Stance, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Mid-stance Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

#combine and publish violin plots
tiff(filename = "Boxplot_Hip.tif", width = 166, height = 166, units = "mm", res = 600, compression = "lzw") #166x166*2 or 166x166*1.5

plot.new()

ggarrange(Box.Hip.Max, 
          Box.Hip.Min, 
          Box.Toe.On, 
          Box.Toe.Off, 
          Box.Mid.Stance, 
          labels = c("A", "B", "C", "D", "E"),
          hjust = -4,
          ncol = 2, nrow = 3)

dev.off()

###MAIN ANALYSIS

#Seperate variables to individual columns (required as outliers are omitted via row deletion)
Hip.Max.Flex.Data <- Hip[,1:2]
Hip.Min.Flex.Data <- Hip[,c(1,3)]
Hip.Toe.On.Data <- Hip[,c(1,4)]
Hip.Toe.Off.Data <- Hip[,c(1,5)]
Hip.Mid.Stance.Data <- Hip[,c(1,6)]

#Omit outliers (currently performed manually)
Hip.Min.Flex.Data <- Hip.Min.Flex.Data[!(row.names(Hip.Min.Flex.Data) %in% c("29")),]

Hip.Max.Flex.Data <- Hip.Max.Flex.Data[!(row.names(Hip.Max.Flex.Data) %in% c("82")),]
Hip.Max.Flex.Data <- Hip.Max.Flex.Data[!(row.names(Hip.Max.Flex.Data) %in% c("90")),]
Hip.Max.Flex.Data <- Hip.Max.Flex.Data[!(row.names(Hip.Max.Flex.Data) %in% c("89")),]
Hip.Max.Flex.Data <- Hip.Max.Flex.Data[!(row.names(Hip.Max.Flex.Data) %in% c("67")),]
Hip.Max.Flex.Data <- Hip.Max.Flex.Data[!(row.names(Hip.Max.Flex.Data) %in% c("62")),]
Hip.Max.Flex.Data <- Hip.Max.Flex.Data[!(row.names(Hip.Max.Flex.Data) %in% c("81")),]

Hip.Toe.On.Data <- Hip.Toe.On.Data[!(row.names(Hip.Toe.On.Data) %in% c("82")),]
Hip.Toe.On.Data <- Hip.Toe.On.Data[!(row.names(Hip.Toe.On.Data) %in% c("90")),]
Hip.Toe.On.Data <- Hip.Toe.On.Data[!(row.names(Hip.Toe.On.Data) %in% c("67")),]

Hip.Toe.Off.Data <- Hip.Toe.Off.Data[!(row.names(Hip.Toe.Off.Data) %in% c("11")),]
Hip.Toe.Off.Data <- Hip.Toe.Off.Data[!(row.names(Hip.Toe.Off.Data) %in% c("10")),]
Hip.Toe.Off.Data <- Hip.Toe.Off.Data[!(row.names(Hip.Toe.Off.Data) %in% c("48")),]
Hip.Toe.Off.Data <- Hip.Toe.Off.Data[!(row.names(Hip.Toe.Off.Data) %in% c("37")),]


#Shapiro-Wilks Test for Normality
res.shap <- shapiro_test(Hip.Min.Flex.Data[,2])
print(res.shap)
res.shap <- shapiro_test(Hip.Max.Flex.Data[,2])
print(res.shap)
res.shap <- shapiro_test(Hip.Toe.On.Data[,2])
print(res.shap)
res.shap <- shapiro_test(Hip.Toe.Off.Data[,2])
print(res.shap)
res.shap <- shapiro_test(Hip.Mid.Stance.Data[,2])
print(res.shap)

#Levene's test for homogeneity of variance
res.lt <- levene_test(Hip.Min.Flex.Data[,2] ~ Breed, data = Hip.Min.Flex.Data)
print(res.lt)
res.lt <- levene_test(Hip.Max.Flex.Data[,2] ~ Breed, data = Hip.Max.Flex.Data)
print(res.lt)
res.lt <- levene_test(Hip.Toe.On.Data[,2] ~ Breed, data = Hip.Toe.On.Data)
print(res.lt)
res.lt <- levene_test(Hip.Toe.Off.Data[,2] ~ Breed, data = Hip.Toe.Off.Data)
print(res.lt)
res.lt <- levene_test(Hip.Mid.Stance.Data[,2] ~ Breed, data = Hip.Mid.Stance.Data)
print(res.lt)

#Kruskal-Wallis Test & output to .xls
KW.Hip.Min.Flex <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Hip.Min.Flex <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Hip.Min.Flex.Data[,2] ~ Breed, data = Hip.Min.Flex.Data)
res.ES <- kruskal_effsize(Hip.Min.Flex.Data[,2] ~ Breed, data = Hip.Min.Flex.Data)
print(res.KW)
print(res.ES)
KW.Hip.Min.Flex[i,] <- data.frame(var = names(Hip.Min.Flex.Data[2]), res.KW, row.names = NULL)
ES.Hip.Min.Flex[i,] <- data.frame(var = names(Hip.Min.Flex.Data[2]), res.ES, row.names = NULL)
colnames(KW.Hip.Min.Flex) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Hip.Min.Flex) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Hip.Min.Flex <- cbind(KW.Hip.Min.Flex, ES.Hip.Min.Flex)
Final.Hip.Min.Flex <- Final.Hip.Min.Flex[ -c(2, 7:10, 12:13) ]
Final.Hip.Min.Flex <- Final.Hip.Min.Flex[ -c(1:5), ]

KW.Hip.Max.Flex <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Hip.Max.Flex <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Hip.Max.Flex.Data[,2] ~ Breed, data = Hip.Max.Flex.Data)
res.ES <- kruskal_effsize(Hip.Max.Flex.Data[,2] ~ Breed, data = Hip.Max.Flex.Data)
print(res.KW)
print(res.ES)
KW.Hip.Max.Flex[i,] <- data.frame(var = names(Hip.Max.Flex.Data[2]), res.KW, row.names = NULL)
ES.Hip.Max.Flex[i,] <- data.frame(var = names(Hip.Max.Flex.Data[2]), res.ES, row.names = NULL)
colnames(KW.Hip.Max.Flex) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Hip.Max.Flex) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Hip.Max.Flex <- cbind(KW.Hip.Max.Flex, ES.Hip.Max.Flex)
Final.Hip.Max.Flex <- Final.Hip.Max.Flex[ -c(2, 7:10, 12:13) ]
Final.Hip.Max.Flex <- Final.Hip.Max.Flex[ -c(1:5), ]

KW.Hip.Toe.On <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Hip.Toe.On <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Hip.Toe.On.Data[,2] ~ Breed, data = Hip.Toe.On.Data)
res.ES <- kruskal_effsize(Hip.Toe.On.Data[,2] ~ Breed, data = Hip.Toe.On.Data)
print(res.KW)
print(res.ES)
KW.Hip.Toe.On[i,] <- data.frame(var = names(Hip.Toe.On.Data[2]), res.KW, row.names = NULL)
ES.Hip.Toe.On[i,] <- data.frame(var = names(Hip.Toe.On.Data[2]), res.ES, row.names = NULL)
colnames(KW.Hip.Toe.On) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Hip.Toe.On) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Hip.Toe.On <- cbind(KW.Hip.Toe.On, ES.Hip.Toe.On)
Final.Hip.Toe.On <- Final.Hip.Toe.On[ -c(2, 7:10, 12:13) ]
Final.Hip.Toe.On <- Final.Hip.Toe.On[ -c(1:5), ]

KW.Hip.Toe.Off <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Hip.Toe.Off <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Hip.Toe.Off.Data[,2] ~ Breed, data = Hip.Toe.Off.Data)
res.ES <- kruskal_effsize(Hip.Toe.Off.Data[,2] ~ Breed, data = Hip.Toe.Off.Data)
print(res.KW)
print(res.ES)
KW.Hip.Toe.Off[i,] <- data.frame(var = names(Hip.Toe.Off.Data[2]), res.KW, row.names = NULL)
ES.Hip.Toe.Off[i,] <- data.frame(var = names(Hip.Toe.Off.Data[2]), res.ES, row.names = NULL)
colnames(KW.Hip.Toe.Off) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Hip.Toe.Off) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Hip.Toe.Off <- cbind(KW.Hip.Toe.Off, ES.Hip.Toe.Off)
Final.Hip.Toe.Off <- Final.Hip.Toe.Off[ -c(2, 7:10, 12:13) ]
Final.Hip.Toe.Off <- Final.Hip.Toe.Off[ -c(1:5), ]

KW.Hip.Mid.Stance <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Hip.Mid.Stance <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Hip.Mid.Stance.Data[,2] ~ Breed, data = Hip.Mid.Stance.Data)
res.ES <- kruskal_effsize(Hip.Mid.Stance.Data[,2] ~ Breed, data = Hip.Mid.Stance.Data)
print(res.KW)
print(res.ES)
KW.Hip.Mid.Stance[i,] <- data.frame(var = names(Hip.Mid.Stance.Data[2]), res.KW, row.names = NULL)
ES.Hip.Mid.Stance[i,] <- data.frame(var = names(Hip.Mid.Stance.Data[2]), res.ES, row.names = NULL)
colnames(KW.Hip.Mid.Stance) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Hip.Mid.Stance) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Hip.Mid.Stance <- cbind(KW.Hip.Mid.Stance, ES.Hip.Mid.Stance)
Final.Hip.Mid.Stance <- Final.Hip.Mid.Stance[ -c(2, 7:10, 12:13) ]
Final.Hip.Mid.Stance <- Final.Hip.Mid.Stance[ -c(1:5), ]

write.xlsx(Final.Hip.Min.Flex, file = "KW_Hip.xlsx", sheetName = "Hip.Min.Flex")
write.xlsx(Final.Hip.Max.Flex, file = "KW_Hip.xlsx", sheetName = "Hip.Max.Flex", append = TRUE)
write.xlsx(Final.Hip.Toe.On, file = "KW_Hip.xlsx", sheetName = "Hip.Toe.On", append = TRUE)
write.xlsx(Final.Hip.Toe.Off, file = "KW_Hip.xlsx", sheetName = "Hip.Toe.Off", append = TRUE)
write.xlsx(Final.Hip.Mid.Stance, file = "KW_Hip.xlsx", sheetName = "Hip.Mid.Stance", append = TRUE)

#Dunn's Test & output to .xls
dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Hip.Min.Flex <- dunn_vector(Hip.Min.Flex.Data, reformulate("Breed", names(Hip.Min.Flex.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Hip.Min.Flex

dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Hip.Max.Flex <- dunn_vector(Hip.Max.Flex.Data, reformulate("Breed", names(Hip.Max.Flex.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Hip.Max.Flex

dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Hip.Toe.On <- dunn_vector(Hip.Toe.On.Data, reformulate("Breed", names(Hip.Toe.On.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Hip.Toe.On

dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Hip.Toe.Off <- dunn_vector(Hip.Toe.Off.Data, reformulate("Breed", names(Hip.Toe.Off.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Hip.Toe.Off

dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Hip.Mid.Stance <- dunn_vector(Hip.Mid.Stance.Data, reformulate("Breed", names(Hip.Mid.Stance.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Hip.Mid.Stance

write.xlsx(DN.Hip.Min.Flex, file = "Dunn_Hip.xlsx", sheetName = "Hip.Min.Flex")
write.xlsx(DN.Hip.Max.Flex, file = "Dunn_Hip.xlsx", sheetName = "Hip.Max.Flex", append = TRUE)
write.xlsx(DN.Hip.Toe.On, file = "Dunn_Hip.xlsx", sheetName = "Hip.Toe.On", append = TRUE)
write.xlsx(DN.Hip.Toe.Off, file = "Dunn_Hip.xlsx", sheetName = "Hip.Toe.Off", append = TRUE)
write.xlsx(DN.Hip.Mid.Stance, file = "Dunn_Hip.xlsx", sheetName = "Hip.Mid.Stance", append = TRUE)

#Violin Plots
Violin.Hip.Min.Flex <- ggplot(Hip.Min.Flex.Data, aes(x=Breed, y=Hip_Min_Flex, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Minimum Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Violin.Hip.Max.Flex <- ggplot(Hip.Max.Flex.Data, aes(x=Breed, y=Hip_Max_Flex, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Maximum Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Violin.Hip.Toe.On <- ggplot(Hip.Toe.On.Data, aes(x=Breed, y=Hip_Toe_on, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Toe-on Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Violin.Hip.Toe.Off <- ggplot(Hip.Toe.Off.Data, aes(x=Breed, y=Hip_Toe_off, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Toe-off Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Violin.Hip.Mid.Stance <- ggplot(Hip.Mid.Stance.Data, aes(x=Breed, y=Hip_Mid_Stance, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Mid-stance Hip Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

#combine and publish violin plots
tiff(filename = "Violin_Hip.tif", width = 166, height = 166, units = "mm", res = 600, compression = "lzw") #166x166*2 or 166x166*1.5

plot.new()

ggarrange(Violin.Hip.Min.Flex, 
          Violin.Hip.Max.Flex, 
          Violin.Hip.Toe.On, 
          Violin.Hip.Toe.Off, 
          Violin.Hip.Mid.Stance, 
          labels = c("A", "B", "C", "D", "E"),
          hjust = -4,
          ncol = 2, nrow = 3)

dev.off()
